#! /usr/bin/perl
#---------------------------------------------------------------------
# $Id: testFFV.pl 4685 2007-01-27 20:57:31Z cjm $
# Copyright 2006 Christopher J. Madsen
#
# Test suite for FFV
#---------------------------------------------------------------------

use strict;
use Test::More tests => 29;
use File::Temp 'tempdir';
use FindBin    '$Bin';

my $dataDir = "$Bin/testdata";
my $ffv = $ENV{FFV_EXE_PATH};
my $Win32 = ($^O =~ /Win32/);

sub exSuccess       () { 0 }
sub exMissing       () { 1 }
sub exMismatch      () { 2 }
sub exIOError       () { 3 }
sub exInternalError () { 4 }
sub exUsage         () { 5 }

if (not $ffv) {
  # Under Windows, we'll look in the build directories:
  if ($Win32) {
    my $debug   = "$Bin/../win32/Debug/FFV.exe";
    my $release = "$Bin/../win32/Release/FFV.exe";

    if (-e $debug) {
      if (-e $release) {
        # Both $debug & $release exist, use the newest one:
        $ffv = ((-M $debug < -M $release) ? $debug : $release);
      } else {
        $ffv = $debug;          # Only $debug exists
      }
    } elsif (-e $release) {
      $ffv = $release;          # Only $release exists
    }
  } # end if Win32

  $ffv ||= 'ffv';               # Assume it's on the path
} # end

print "# Testing $ffv\n";

#=====================================================================
# This is a modified copy of mt19937ar-cok.c
# from http://www-personal.engin.umich.edu/~wagnerr/Mersenne-1.0.zip
#
# I just removed the functions I didn't need.

use Inline C => <<'__END C__';
/*
   A C-program for MT19937, with initialization improved 2002/2/10.
   Coded by Takuji Nishimura and Makoto Matsumoto.
   This is a faster version by taking Shawn Cokus's optimization,
   Matthe Bellew's simplification, Isaku Wada's real version.

   Before using, initialize the state by using init_genrand(seed)
   or init_by_array(init_key, key_length).

   Copyright (C) 1997 - 2002, Makoto Matsumoto and Takuji Nishimura,
   All rights reserved.

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

     1. Redistributions of source code must retain the above copyright
        notice, this list of conditions and the following disclaimer.

     2. Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.

     3. The names of its contributors may not be used to endorse or promote
        products derived from this software without specific prior written
        permission.

   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
   A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR
   CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
   EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
   PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
   PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
   LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
   NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
   SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


   Any feedback is very welcome.
   http://www.math.keio.ac.jp/matumoto/emt.html
   email: matumoto@math.keio.ac.jp
*/

/* Period parameters */
#define N 624
#define M 397
#define MATRIX_A 0x9908b0dfUL   /* constant vector a */
#define UMASK 0x80000000UL /* most significant w-r bits */
#define LMASK 0x7fffffffUL /* least significant r bits */
#define MIXBITS(u,v) ( ((u) & UMASK) | ((v) & LMASK) )
#define TWIST(u,v) ((MIXBITS(u,v) >> 1) ^ ((v)&1UL ? MATRIX_A : 0UL))

static unsigned long state[N]; /* the array for the state vector  */
static int left = 1;
static int initf = 0;
static unsigned long *next;

/* initializes state[N] with a seed */
void init_genrand(unsigned long s)
{
    int j;
    state[0]= s & 0xffffffffUL;
    for (j=1; j<N; j++) {
        state[j] = (1812433253UL * (state[j-1] ^ (state[j-1] >> 30)) + j);
        /* See Knuth TAOCP Vol2. 3rd Ed. P.106 for multiplier. */
        /* In the previous versions, MSBs of the seed affect   */
        /* only MSBs of the array state[].                        */
        /* 2002/01/09 modified by Makoto Matsumoto             */
        state[j] &= 0xffffffffUL;  /* for >32 bit machines */
    }
    left = 1; initf = 1;
}

static void next_state(void)
{
    unsigned long *p=state;
    int j;

    /* if init_genrand() has not been called, */
    /* a default initial seed is used         */
    if (initf==0) init_genrand(5489UL);

    left = N;
    next = state;

    for (j=N-M+1; --j; p++)
        *p = p[M] ^ TWIST(p[0], p[1]);

    for (j=M; --j; p++)
        *p = p[M-N] ^ TWIST(p[0], p[1]);

    *p = p[M-N] ^ TWIST(p[0], state[0]);
}

/* generates a random number on [0,0xffffffff]-interval */
unsigned long genrand_int32()
{
    unsigned long y;

    if (--left == 0) next_state();
    y = *next++;

    /* Tempering */
    y ^= (y >> 11);
    y ^= (y << 7) & 0x9d2c5680UL;
    y ^= (y << 15) & 0xefc60000UL;
    y ^= (y >> 18);

    return y;
}

__END C__
#=====================================================================
# Subroutines:
#---------------------------------------------------------------------
sub file
{
  my ($file) = @_;

  local $/ = undef;

  unless(open(FILE, '<', $file)) {
    warn "Can't open $file: $!";
    return undef;
  }
  my $contents = <FILE>;
  close FILE;

  return $contents;
} # end file

#---------------------------------------------------------------------
# Convert backslashes to forward slashes on Win32:

sub fs
{
  $_[0] =~ s!\\!/!g if $Win32;
  $_[0];
} # end fs

#---------------------------------------------------------------------
sub genfile
{
  my ($file, $seed, $length) = @_;

  # Create the output file:
  open(OUT, '>', $file) or die "Can't open $file: $!";
  binmode OUT;

  # Initialize the random number generator:
  init_genrand($seed);
  my $buf;

  # Write the main body of the file, 256 bytes at a time:
  while (1) {
    $buf = pack('V64', map { genrand_int32() } (1 .. 64));

    last if $length < 256;
    $length -= 256;

    print OUT $buf or die "Can't write $file: $!";
  } # end loop

  # If the length isn't a multiple of 256, write the final chunk:
  if ($length > 0) {
    substr($buf, $length) = '';
    print OUT $buf or die "Can't write $file: $!";
  } # end if length is not a multiple of 256

  close OUT;
} # end genfile

#---------------------------------------------------------------------
sub runTest
{
  my ($testName, $args, $status, $expectedOutput) = @_;

  my $output = `$ffv $args 2>&1`;

  is($? >> 8, $status, "$testName status");

  $output =~ s/ *\r?\n/\n/mg;   # Clean up trailing whitespace
  fs($output);                  # Convert backslashes

  is($output, $expectedOutput, "$testName output");
} # end runTest

#=====================================================================
# Begin testing:
#---------------------------------------------------------------------
# Get version information from configure.ac:

our ($name, $major, $minor);
do "$Bin/getversion.pm";

my $version = "$major.$minor";

#---------------------------------------------------------------------
# Create directory and files:

my $dir = tempdir(CLEANUP => 1);
#my $dir="/tmp/testFFV"; system "mkdir $dir || rm -rf $dir/*"; # For development
chdir $dir or die "Failed to change to $dir";

#genfile('junk.out', 57, 510);
genfile('file1.bin', 57, 100 * 1024);
genfile('file2.out', 42, 99999);

#---------------------------------------------------------------------
runTest('Created check.md5', '-m check.md5', exSuccess,
        <<"END Created check.md5");

Fast File Validator $version by Christopher J. Madsen
Found 2 files

Creating MD5 file check.md5:
1/2 file1.bin
2/2 file2.out
Added 2 files to check.md5
END Created check.md5

#---------------------------------------------------------------------
is(file('check.md5'), <<'END Contents of check.md5', 'Contents of check.md5');
MD5(file1.bin) = 8b9330c4cb7fc5b6c4f5af8f8ef9512c
MD5(file2.out) = 917fc62624ecb8b407c8ca693635d99c
END Contents of check.md5

#---------------------------------------------------------------------
runTest('Verified check.md5', '', exSuccess, <<"END Verified check.md5");

Fast File Validator $version by Christopher J. Madsen

check.md5:
1/2 file1.bin ............................................................. OK
2/2 file2.out ............................................................. OK
2 files checked, all ok!
END Verified check.md5

#---------------------------------------------------------------------
genfile('file2.out', 42, 100000);

runTest('Changed file2.out', '', exMismatch, <<"END Changed file2.out");

Fast File Validator $version by Christopher J. Madsen

check.md5:
1/2 file1.bin ............................................................. OK
2/2 file2.out .............................................................BAD
2 files checked
1 file different
END Changed file2.out

#---------------------------------------------------------------------
runTest('Created check.sfv', '-s check.sfv file*', exSuccess,
        <<"END Created check.sfv");

Fast File Validator $version by Christopher J. Madsen
Found 2 files

Creating SFV file check.sfv:
1/2 file1.bin
2/2 file2.out
Added 2 files to check.sfv
END Created check.sfv

#---------------------------------------------------------------------
is(file('check.sfv'), <<"END Contents of check.sfv", 'Contents of check.sfv');
; Generated by Fast File Validator $version
;
; Compatibility hack for those poor misguided souls using WIN-SFV32:
; Generated by WIN-SFV32 v1
;
file1.bin FDD6BF96
file2.out 69779FF5
END Contents of check.sfv

#---------------------------------------------------------------------
runTest('Verified check.sfv', 'check.sfv', exSuccess,
        <<"END Verified check.sfv");

Fast File Validator $version by Christopher J. Madsen

check.sfv:
1/2 file1.bin ............................................................. OK
2/2 file2.out ............................................................. OK
2 files checked, all ok!
END Verified check.sfv

#---------------------------------------------------------------------
runTest('Verified test1.md5', "$dataDir/test1.md5", exSuccess,
        <<"END Verified test1.md5");

Fast File Validator $version by Christopher J. Madsen

$dataDir/test1.md5:
1/2 file1.bin ............................................................. OK
2/2 file2.out ............................................................. OK
2 files checked, all ok!
END Verified test1.md5

#---------------------------------------------------------------------
runTest('Verified test1.sfv', "$dataDir/test1.sfv", exSuccess,
        <<"END Verified test1.sfv");

Fast File Validator $version by Christopher J. Madsen

$dataDir/test1.sfv:
1/2 file1.bin ............................................................. OK
2/2 file2.out ............................................................. OK
2 files checked, all ok!
END Verified test1.sfv

#=====================================================================
mkdir 'foo';
mkdir 'foo/bar';
mkdir 'baz';

genfile('foo/alpha.bin',       77,  8472);
genfile('foo/beta.pnf',        94, 32764);
genfile('foo/gamma.bin',       27, 98345);
genfile('foo/bar/delta.bin',   97,  3456);
genfile('foo/bar/epsilon.bin', 97,  1234);
genfile('baz/zeta.bin',        64,  5189);
genfile('baz/eta.bin',        127, 88777);
genfile('baz/theta.bin',      193, 19876);

#---------------------------------------------------------------------
runTest('Created subdirs.md5', '-m subdirs.md5 *.bin *.out foo baz',
        exSuccess, <<"END Created subdirs.md5");

Fast File Validator $version by Christopher J. Madsen
Found 10 files

Creating MD5 file subdirs.md5:
 1/10 baz/eta.bin
 2/10 baz/theta.bin
 3/10 baz/zeta.bin
 4/10 file1.bin
 5/10 file2.out
 6/10 foo/alpha.bin
 7/10 foo/bar/delta.bin
 8/10 foo/bar/epsilon.bin
 9/10 foo/beta.pnf
10/10 foo/gamma.bin
Added 10 files to subdirs.md5
END Created subdirs.md5

#---------------------------------------------------------------------
is(fs(file('subdirs.md5')),
   <<'END Contents of subdirs.md5', 'Contents of subdirs.md5');
MD5(baz/eta.bin) = 3c9d890b5cabc59fe0d13b4c696ad6bd
MD5(baz/theta.bin) = 6adc5c5362b838228ad9c539125606be
MD5(baz/zeta.bin) = c6b36c964e667f8756882374d3f38515
MD5(file1.bin) = 8b9330c4cb7fc5b6c4f5af8f8ef9512c
MD5(file2.out) = bda8c50024a2fb4ac8467b21d719ad0a
MD5(foo/alpha.bin) = dbb831bc5551ac3b68bc47aed2fe8138
MD5(foo/bar/delta.bin) = 03394b0dcf163b15290438750958aa49
MD5(foo/bar/epsilon.bin) = d2b1113d7c67bd82076880c79076efba
MD5(foo/beta.pnf) = c284495f7a9734fd116eb34b5e60a28e
MD5(foo/gamma.bin) = 9fc5b27fc87f63782909a41e1bfd7c20
END Contents of subdirs.md5

#---------------------------------------------------------------------
chdir 'foo';

runTest('Created foo.md5', '-m foo.md5', exSuccess,
        <<"END Created foo.md5");

Fast File Validator $version by Christopher J. Madsen
Found 5 files

Creating MD5 file foo.md5:
1/5 alpha.bin
2/5 bar/delta.bin
3/5 bar/epsilon.bin
4/5 beta.pnf
5/5 gamma.bin
Added 5 files to foo.md5
END Created foo.md5

#---------------------------------------------------------------------
chdir '../baz';

runTest('Created baz.md5', '-m baz.md5', exSuccess,
        <<"END Created baz.md5");

Fast File Validator $version by Christopher J. Madsen
Found 3 files

Creating MD5 file baz.md5:
1/3 eta.bin
2/3 theta.bin
3/3 zeta.bin
Added 3 files to baz.md5
END Created baz.md5

#---------------------------------------------------------------------
chdir '..';

runTest('Verified non-rel', 'subdirs.md5 foo/foo.md5 baz/baz.md5',
        exMissing, <<"END Verified non-rel");

Fast File Validator $version by Christopher J. Madsen

[1/3] subdirs.md5:
 1/10 baz/eta.bin ......................................................... OK
 2/10 baz/theta.bin ....................................................... OK
 3/10 baz/zeta.bin ........................................................ OK
 4/10 file1.bin ........................................................... OK
 5/10 file2.out ........................................................... OK
 6/10 foo/alpha.bin ....................................................... OK
 7/10 foo/bar/delta.bin ................................................... OK
 8/10 foo/bar/epsilon.bin ................................................. OK
 9/10 foo/beta.pnf ........................................................ OK
10/10 foo/gamma.bin ....................................................... OK
10 files checked, all ok!

[2/3] foo/foo.md5:
1/5 alpha.bin .............................................................MISS
2/5 bar/delta.bin .........................................................MISS
3/5 bar/epsilon.bin .......................................................MISS
4/5 beta.pnf ..............................................................MISS
5/5 gamma.bin .............................................................MISS
5 files checked
5 files missing

[3/3] baz/baz.md5:
1/3 eta.bin ...............................................................MISS
2/3 theta.bin .............................................................MISS
3/3 zeta.bin ..............................................................MISS
3 files checked
3 files missing

18 files checked in 3 SFV files, found errors in 2 SFV files
 8 files missing
END Verified non-rel

#---------------------------------------------------------------------
runTest('Verified --relative', '-r subdirs.md5 foo/foo.md5 baz/baz.md5',
        exSuccess, <<"END Verified --relative");

Fast File Validator $version by Christopher J. Madsen

[1/3] subdirs.md5:
 1/10 baz/eta.bin ......................................................... OK
 2/10 baz/theta.bin ....................................................... OK
 3/10 baz/zeta.bin ........................................................ OK
 4/10 file1.bin ........................................................... OK
 5/10 file2.out ........................................................... OK
 6/10 foo/alpha.bin ....................................................... OK
 7/10 foo/bar/delta.bin ................................................... OK
 8/10 foo/bar/epsilon.bin ................................................. OK
 9/10 foo/beta.pnf ........................................................ OK
10/10 foo/gamma.bin ....................................................... OK
10 files checked, all ok!

[2/3] foo/foo.md5:
1/5 foo/alpha.bin ......................................................... OK
2/5 foo/bar/delta.bin ..................................................... OK
3/5 foo/bar/epsilon.bin ................................................... OK
4/5 foo/beta.pnf .......................................................... OK
5/5 foo/gamma.bin ......................................................... OK
5 files checked, all ok!

[3/3] baz/baz.md5:
1/3 baz/eta.bin ........................................................... OK
2/3 baz/theta.bin ......................................................... OK
3/3 baz/zeta.bin .......................................................... OK
3 files checked, all ok!

18 files checked in 3 SFV files, all ok!
END Verified --relative

#---------------------------------------------------------------------
my $usage = <<"END --no-such-option";
--no-such-option is not a recognized option

Fast File Validator $version by Christopher J. Madsen

Usage: FFV [options] [FILE ...]
  -a, --append          Append to existing checksum file (when creating one)
  -o, --overwrite       Replace an existing checksum file (when creating one)
  -e, --allow-errors    Don't stop if there's an error reading a file
  -h, --hidden          Include hidden files when creating a checksum file
  -m, --create-md5=FILE Create an MD5 checksum file
  -s, --create-sfv=FILE Create a SFV (CRC-32) file
  -n, --no-verify       Just check if files exist (don't verify checksums)
  -p, --pause           Wait for keypress before exiting
  -r, --relative        Interpret paths relative to the checksum file
] -w, --allow-writing   Process files that are being written by other programs
  -?, --help            Display this help message
      --license         Display license information
      --version         Display version information

Note: You can't combine the --create-sfv and --create-md5 options
      (or the --append and --overwrite options, either).
END --no-such-option

if ($Win32) {
  $usage =~ s/^\]/ /mg;
} else {
  $usage =~ s/^\].+\n//mg;      # Remove Win32-only options
}

runTest('--no-such-option', '--no-such-option', exUsage, $usage);

#---------------------------------------------------------------------
chdir '..';   # Otherwise Windows can't remove the temporary directory
