//--------------------------------------------------------------------
// $Id: FFV.cpp 4684 2007-01-27 20:46:06Z cjm $
//--------------------------------------------------------------------
//
//   Fast File Validator
//   Copyright 2000 by Christopher J. Madsen
//
//   Simple & fast file validation: SFV(CRC-32) & MD5
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License as
//   published by the Free Software Foundation; either version 2 of
//   the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//--------------------------------------------------------------------

#include "FFV.hpp"

const int pci = 0x100; // Number of completion increments per percent
const int MD5hexlen = 32; // Size of MD5 digest w/o trailing null
const int CRChexlen = 8;  // Size of CRC32 w/o trailing null

const char *const  hexDigits = "0123456789ABCDEFabcdef";
const char titleString[] =
  "\nFast File Validator " FFV_VERSION " by Christopher J. Madsen";

void usage(bool showHelp=true, int exitStatus=0);

GetOpt::Found  includeHidden, relativePaths;

#ifdef _WIN32
#include "win32/misc.cpp"
#else
#include "posix/misc.cpp"
#endif

//====================================================================
// Miscellaneous Functions:
//--------------------------------------------------------------------
// Calculate the number of digits in a number:
//
// Input:
//   value:  A non-negative integer
//
// Returns:
//   The number of digits in value

int calcWidth(int value)
{
  int width = 1;
  while (value /= 10)
    ++width;

  return width;
} // end calcWidth

//--------------------------------------------------------------------
// Exit, possibly waiting for a keypress:
//
// Input:
//   status:  The status code to pass to exit()

GetOpt::Found  pauseAtQuit;

void exitNow(int status)
{
  if (pauseAtQuit) {
    fputs("Press the space bar to exit...", stderr);
    fflush(stderr);
    getKey();
  } // end if pause before quitting

  exit(status);
} // end exitNow

//--------------------------------------------------------------------
// Print a filename and error message:
//
// Input:
//   file:   The filename to print with the error message

void errorPrint(const char* file="FFV")
{
#ifdef _WIN32
  DWORD  err = GetLastError();
  TCHAR  buf[512];
  FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | 65, NULL, err, 0,
                buf, sizeof(buf), NULL);
  fprintf(stderr, "%s: %s\n", file, buf);
#else
  perror(file);
#endif
} // end errorPrint

//--------------------------------------------------------------------
// Exit with an error message:
//
// Input:
//   file:   The filename to print with the error message
//   error:  The status code to pass to exit()

void errorExit(const char* file="FFV", int error=exInternalError)
{
  errorPrint(file);
  exitNow(error);
} // end errorExit

//--------------------------------------------------------------------
// Finalize a MD5 digest and convert it to hex:
//
// Input:
//   digest:  Points to the buffer where the digest should be stored
//   md5:     The MD5 object containing the digest
//
// Output:
//   digest:  Contains the digest in hex format
//   md5:     Ready to begin processing a new file

void hexdigest(char* digest, MD5& md5)
{
  BYTE  temp[MD5DigestLen];

  md5.finalize(temp);           // Also reinitializes md5
  for (int i = 0; i < MD5DigestLen; i++) {
    sprintf(digest, "%02x", temp[i]);
    digest += 2;
  }
} // end hexdigest

//--------------------------------------------------------------------
// Decide if a path is relative or absolute:
//
// Input:
//   pathname:  The pathname to check
//
// Returns:
//   true:   The pathname is relative
//   false:  The pathname is absolute

bool isRelativePath(const String& pathname)
{
  // FIXME: handle drive letters?
  return strchr(dirChars, pathname[0]) == NULL;
} // end isRelativePath

//--------------------------------------------------------------------
// Decide if a character is a space or tab:

inline bool isSpace(char c)
{
  return (c == ' ') || (c == '\t');
} // end isSpace

//--------------------------------------------------------------------
// Print an error message (indented 4 spaces):

void printError(errorT err)
{
#ifdef _WIN32
  TCHAR  buf[512];
  FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | 62, NULL, err, 0,
                buf, sizeof(buf), NULL);
  char* s = buf;
#else
  char* s = strerror(err);
#endif

  while (s) {
    char* e = strchr(s, '\n');
    if (e) *(e++) = '\0';
    printf("    %s\n", s);
    s = e;
  } // end while more
} // end printError

//--------------------------------------------------------------------
// Print an error message for the last error (indented 4 spaces):

void printError()
{
#ifdef _WIN32
  printError(GetLastError());
#else
  printError(errno);
#endif
} // end printError

//--------------------------------------------------------------------
// Compare two strings (case insensitive):
//
// Returns:  true iff a < b

bool strILT(const String& a, const String& b)
{
  return (strcasecmp(a.c_str(), b.c_str()) < 0);
} // end strILT

//====================================================================
// Variables & classes:
//--------------------------------------------------------------------
const off_t bufferSize = 32 * 1024;
typedef BYTE  DiskBuffer[bufferSize];

#ifdef _WIN32
DiskBuffer  *buffer;
int    buflen[2];
DWORD  shareMode = FILE_SHARE_READ;
char   outputBuf[MD5hexlen+1];
bool   wantCRC = false;

HANDLE  bufFull[2];
HANDLE  bufEmpty[2];
HANDLE  crcReady;
/// crcUsed is not necessary, because the main loop doesn't start
/// reading the next file until it has printed the result of the
/// previous one.  If this is changed, you must uncomment the lines
/// that concern crcUsed.
///HANDLE  crcUsed;
#endif // _WIN32

bool   wantProgress = false;
bool   stopOnError = true;

enum CountItem // The things we count while verifying files
{
  cntFiles,      // The number of files checked
  cntIOError,    // The number of files we couldn't open
  cntMismatch,   // The number of files that had different contents
  cntMissing,    // The number of files that didn't exist
  cntArraySize   // The number of things being counted
}; // end CountItem

const char // These are the checksum types
  checkCRC32 = 0x01,
  checkMD5   = 0x02;

// This is the length of each checksum type (indexed as [type-1])
const int checkLen[] = { CRChexlen, MD5hexlen };

enum IfExist // Actions to take if the output file exists
{
  iePrompt = 0,
  ieAppend,
  ieOverwrite
}; // end IfExist

//--------------------------------------------------------------------
class FileList
{
 protected:
  StrVec       fileList;
  SVConstItr   curFile;
  const char*  curFileName;
  int          curFileNum;
  int          curFileWidth;
  errorT       curError;

 public:
  FileList() : curFileName(NULL),curFileNum(0),curFileWidth(0),curError(0) {};
  virtual ~FileList() {};
  virtual void  setResult(const char* digest) = 0;
  virtual int   getExitStatus() const = 0;
  virtual bool  getFile(fileT& file, int& step, int& interval,
                        bool& wantCRC) = 0;
  virtual void  printResult(bool showTotals=true) = 0;

  errorT      currentError() const { return curError; };
  const char* currentFile() const { return curFileName; };
  void        recordError();

 protected:
  void  calcInterval(fileT file, int& step, int& interval) const;
  void  calcFileWidth();
  void  printFile(const char* filename);
}; // end FileList

//--------------------------------------------------------------------
class CreateFileList : public FileList
{
  FILE*        outFile;
#ifndef _WIN32
  dev_t        outFileDev;
  ino_t        outFileInode;
#endif
  const char*  outFileName;
  bool         usingCRC;
  bool         started;
  int          errors;

 public:
  CreateFileList(const char* filename, IfExist ifExist, bool useCRC,
                 int argc, char** argv);

  // Overloaded functions:
  void  setResult(const char* digest);
  int   getExitStatus() const;
  bool  getFile(fileT& file, int& step, int& interval, bool& wantCRC);
  void  printResult(bool showTotals=true);
}; // end CreateFileList

//--------------------------------------------------------------------
class VerifyFileList : public FileList
{
 protected:
  StrVec  sfvList;
  SVConstItr  curSfv;
  int   cnt[cntArraySize];
  int   total[cntArraySize];
  int   sfvBadCount, sfvCount;

 public:
  VerifyFileList(int argc, char** argv);
  // Overloaded functions:
  void  setResult(const char* digest);
  int   getExitStatus() const;
  bool  getFile(fileT& file, int& step, int& interval, bool& wantCRC);
  void  printResult(bool showTotals=true);

 private:
  bool  openNextSFV(bool first);
  void  printResult(const char* format, int width, int count) const;
}; // end VerifyFileList

//====================================================================
// Class FileList:
//
// This is the base class for holding the list of files to be processed.
//
// Member Variables:
//   fileList:      For use by the derived classes
//   curError:      The error code from processing the current file
//   curFile:       For use by the derived classes
//   curFileNum:    The number of the file being processed
//   curFileWidth:  The field width for the file number
//--------------------------------------------------------------------
// Calculate how we should update the progress meter:
//
// Input:
//   file:  The file being processed
//
// Output:
//   step:      The value to increment the progress counter by
//   interval:  The number of reads between progress updates

void FileList::calcInterval(fileT file, int& step, int& interval) const
{
#ifdef _WIN32
  DWORD sizeHi;
  ULONGLONG size = GetFileSize(file, &sizeHi);
  size += ULONGLONG(sizeHi) << 32;
#else
  struct stat  s;
  fstat(file, &s);
  const off64_t size = s.st_size;
#endif

  if (size <= bufferSize) {
    interval = 1;
    step = 100 * pci;
  } else {
    interval = max(0x80000 / bufferSize, size / (bufferSize * 100));
    step = int((100.0 * pci * interval) / (size / bufferSize + interval-1));
  }
} // end FileList::calcInterval

//--------------------------------------------------------------------
// Reset the current file number:
//
// Output Variables:
//   curFileNum, curFileWidth

void FileList::calcFileWidth()
{
  curFileNum   = 0;
  curFileWidth = calcWidth(fileList.size());
} // end FileList::calcFileWidth

//--------------------------------------------------------------------
// Print the name and other info for the file to be processed:
//
// Input:
//   filename:  The name of the file
//
// Input Variables:
//   wantProgress, curFileWidth, curFileNum, fileList
//
// Output Variables:
//   curFileName, curFileNum

void FileList::printFile(const char* filename)
{
  curFileName = filename;

  printf(wantProgress ? "  0%% %*d/%d %.*s\r" : "%*d/%d %.*s ",
         curFileWidth, ++curFileNum, fileList.size(),
         72 - 2*curFileWidth, filename);
} // end FileList::printFile

//--------------------------------------------------------------------
// Record the current error for setResult:
//
// Output Variables:
//   curError

void FileList::recordError()
{
#ifdef _WIN32
  curError = GetLastError();
#else
  curError = errno;
#endif
} // end FileList::recordError

//====================================================================
// Prompt to determine action when output file exists:
//
// Input:
//   filename:  The name of the file
//
// Returns:
//   ieAppend:     Append to the existing file
//   ieOverwrite:  Replace the existing file
//   Exits immediately if neither is selected

IfExist promptOverwrite(const char* filename)
{
  fprintf(stderr, "Output file %s already exists\n", filename);

  if (!isatty(fileno(stdin))) exitNow(exUsage);

  fputs("Append, Overwrite, or Cancel (A/O/C): ", stderr);
  fflush(stderr);

  const int c = getKey();

  if ((c > ' ') && (c < 0x7F)) fputc(c, stderr);
  fputs("\n\n", stderr);

  switch (c) {
   case 'a':  case 'A':  return ieAppend;
   case 'o':  case 'O':  return ieOverwrite;
  } // end switch character

  exitNow(exUsage);                // Cancel
  return iePrompt;                 // Never happens
} // end promptOverwrite

//====================================================================
// Class CreateFileList:
//
// This is the class used for creating a SFV or MD5 file
//
// Member Variables:
//   fileList:     Lists the filenames to include in the output
//   curFile:      The filename currently being processed
//   outFileName:  The name of the file we're creating
//   outFile:      The file we're writing to
//   usingCRC:     True for SFV, false for MD5
//   started:      True means curFile has been initialized
//--------------------------------------------------------------------
// Constructor:
//
// Builds the list of files to process and opens the output file.
//
// Input:
//   filename:  Name of the file to create
//   ifExist:   What to do if the output file exists
//   useCRC:    True to create SFV file, false for MD5
//   argc, argv:
//     The files to include in the checksum file (may include wildcards)
//     Note: argv[0] may not be the executable name

CreateFileList::CreateFileList(const char* filename, IfExist ifExist,
                               bool useCRC, int argc, char** argv)
: outFile(NULL),
  outFileName(filename),
  usingCRC(useCRC),
  started(false),
  errors(0)
{
#ifdef _WIN32
  WIN32_FIND_DATA fd;
  String          dir;
  StrVec          dirList;
  StrVec          dirQueue;

  if (argc < 2)
    dirQueue.push_back("*");
  else for (int argi = 1; argi < argc; ++argi) {
    String s(argv[argi]);
    if (!s.empty()) {
      if (strchr(dirChars, *(s.end() - 1)))
        s += '*';
      dirQueue.push_back(s);
    }
  } // end for command line arguments

  // Note: We can't just append to dirList, because vector::push_back
  // invalidates the iterator.  So we add new directories to dirQueue
  // instead.  Then, when we've finished with dirList, we see if there
  // are new entries in dirQueue.

  while (!dirQueue.empty()) {
    dirList.clear();
    dirList.swap(dirQueue);
    for (SVConstItr fn = dirList.begin(); fn != dirList.end(); ++fn) {
      // Remember the path, because FindFirstFile won't:
      dir.assign(*fn, 0, dirLength(fn->c_str()));

      HANDLE  fh = FindFirstFile(fn->c_str(), &fd);
      if (fh == INVALID_HANDLE_VALUE) {
        if (stopOnError) errorExit(fn->c_str(), exInternalError);
        else             errorPrint(fn->c_str());
      } else {
        do {
          if (((fd.dwFileAttributes & FILE_ATTRIBUTE_HIDDEN) == 0) ||
              includeHidden) {
            String  s(dir);
            s += fd.cFileName;
            if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
              if (strcmp(".", fd.cFileName) && strcmp("..", fd.cFileName))
                dirQueue.push_back(s += "\\*");
            } else if (!filename || stricmp(s.c_str(), filename))
              fileList.push_back(s);
          } // end if not hidden or directory
        } while (FindNextFile(fh, &fd));
        if (GetLastError() != ERROR_NO_MORE_FILES) {
          if (stopOnError) errorExit();
          else             printError();
        }
        FindClose(fh);
      }
    } // end for each element in dirList
  } // end while dirQueue isn't empty
#else
  collectInto = &fileList;
  collectIgnore = filename;

  if (argc < 2)
    nftw(".", nftwCollectFunc, 16, FTW_ACTIONRETVAL);
  else for (int argi = 1; argi < argc; ++argi) {
    if (argv[argi][0])
      nftw(argv[argi], nftwCollectFunc, 16, FTW_ACTIONRETVAL);
  } // end for command line arguments
#endif

  if (!fileList.empty()) {
    printf("Found %d files\n\n", fileList.size());
    calcFileWidth();
    sort(fileList.begin(), fileList.end(), strILT);

#ifdef _WIN32
    if ((ifExist == iePrompt) &&
        (GetFileAttributes(filename) != 0xFFFFFFFF))
      ifExist = promptOverwrite(filename);
#else
    if (ifExist == iePrompt) {
      struct stat s;
      if (lstat(filename, &s) == 0)
        ifExist = promptOverwrite(filename);
    }
#endif
    const bool append = (ifExist == ieAppend);

    outFile = fopen(filename, append ? "a" : "w");
    if (!outFile) errorExit(filename, exIOError);
    printf("%s %s file %s:\n",
           (append ? "Appending to" : "Creating"),
           (usingCRC ? "SFV" : "MD5"),
           filename);

#ifndef _WIN32
    struct stat s;
    fstat(fileno(outFile), &s);
    outFileDev   = s.st_dev;
    outFileInode = s.st_ino;
#endif

    if (usingCRC && !append) {
      fputs("; Generated by Fast File Validator " FFV_VERSION "\n;\n; "
            "Compatibility hack for those poor misguided souls using WIN-SFV32:"
            "\n; Generated by WIN-SFV32 v1\n;\n",
            outFile);
    } // end if creating a new SFV file
  } // end if found files
} // end CreateFileList::CreateFileList

//--------------------------------------------------------------------
// Write the current file's checksum to the output:
//
// Input:
//   digest:  The checksum for the file

void CreateFileList::setResult(const char* digest)
{
  if (usingCRC)
    fprintf(outFile, "%s %s\n", curFile->c_str(), digest);
  else
    fprintf(outFile, "MD5(%s) = %s\n", curFile->c_str(), digest);

  putchar('\n');                // Begin new line in progress report

  if (curError) {
    printError(curError);
    curError = 0;
  }
} // end CreateFileList::setResult

//--------------------------------------------------------------------
// Summarize the result for our exit status:

int CreateFileList::getExitStatus() const
{
  return errors ? exMissing : exSuccess;
} // end CreateFileList::getExitStatus

//--------------------------------------------------------------------
// Open the next file to be added to the output:
//
// Output:
//   file:      The file to process
//   step:      The value to increment the progress counter by
//   interval:  The number of reads between progress updates
//   wantCRC:   True to calculate CRC32, false to calculate MD5
//
// Returns:
//   true:   Opened a new file
//   false:  No more files to process

bool CreateFileList::getFile(fileT& file, int& step, int& interval,
                             bool& wantCRC)
{
 retry:
  if (started)
    ++curFile;
  else {
    curFile = fileList.begin();
    started = true;
  }

  if (curFile == fileList.end()) return false;

  printFile(curFile->c_str());

#ifdef _WIN32
  file = CreateFile(
    curFile->c_str(), GENERIC_READ, shareMode, NULL,
    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL
  );
#else
  file = open(curFile->c_str(), O_RDONLY);
#endif

  if (file == invalidFile) {
    ++errors;
    puts("ERR ");
    printError();
    goto retry;
  }

#ifndef _WIN32
  struct stat s;
  fstat(file, &s);
  if (outFileDev == s.st_dev && outFileInode == s.st_ino) {
    close(file);
    ++errors;
    puts("SKIP\n  You can't include the output file!");
    goto retry;
  }
#endif

  calcInterval(file, step, interval);
  wantCRC = usingCRC;

  return true;
} // end CreateFileList::getFile

//--------------------------------------------------------------------
// Print final results:

void CreateFileList::printResult(bool)
{
  if (outFile) {
    fclose(outFile);
    outFile = NULL;
  }

  const int added = fileList.size() - errors;
  if (added)
    printf("Added %d file%s to %s\n", added, (added == 1 ? "" : "s"),
           outFileName);
  if (errors)
    printf("%d file%s could not be opened\n",
           errors, (errors == 1 ? "" : "s"));
} // end CreateFileList::printResult

//====================================================================
// Class VerifyFileList:
//
// This is the class used when verifying files.
//
// Member Variables:
//   fileList:
//     Lists the files in the current SFV file.  Each entry consists
//     of one character indicating the checksum type (checkCRC32 or
//     checkMD5), followed by the checksum and the filename.  The
//     length of the checksum is determined by the checksum type.
//   curFile:
//     The file currently being processed
//   sfvList:
//     Lists the filenames of the SFV files to check
//   curSfv:
//     The SFV file currently being processed
//   cnt:
//     Totals for the current SFV file
//     (see CountItem for what gets counted)
//   total:
//     Totals for all SFV files
//     (see CountItem for what gets counted)
//   sfvCount:
//     The number of SFV files processed
//   sfvBadCount:
//     The number of SFV files which had errors
//--------------------------------------------------------------------
// Constructor:
//
// Builds the list of SFV files.
//
// Input:
//   argc, argv:
//     The SFV files to check (may include wildcards)
//     Note: argv[0] may not be the executable name

VerifyFileList::VerifyFileList(int argc, char** argv)
: sfvBadCount(0),
  sfvCount(0)
{
  memset(cnt, 0, sizeof(cnt));
  memset(total, 0, sizeof(total));

  //..................................................................
#ifdef _WIN32
  bool  fatalErrors = true;

  if (argc < 2) {
    static char *defaultArgs[] = { NULL, "*.sfv", "*.md5" };
    argc = 3;
    argv = defaultArgs;
    fatalErrors = false;
  }

  String          dir;
  WIN32_FIND_DATA fd;

  for (int argi = 1; argi < argc; ++argi) {
    HANDLE  fh = FindFirstFile(argv[argi], &fd);
    if (fh == INVALID_HANDLE_VALUE) {
      if (fatalErrors) errorExit(argv[argi], exUsage);
      continue;                 // Just skip this one
    }
    dir.assign(argv[argi], dirLength(argv[argi]));
    do {
      if (fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) {
        fprintf(stderr, "%s is a directory\n", argv[argi]);
        exitNow(exUsage);
      } // end if a directory
      String  s(dir);
      s += fd.cFileName;
      sfvList.push_back(s);
    } while (FindNextFile(fh, &fd));
  } // end for argv
  // Continues with common code...

  //..................................................................
#else // POSIX
  if (argc < 2) {
    glob_t gt;
    memset(&gt, 0, sizeof(gt));

    int result = glob("*.sfv", GLOB_MARK, NULL, &gt);
    if (!result || result == GLOB_NOMATCH)
      result =  glob("*.md5", GLOB_MARK | GLOB_APPEND, NULL, &gt);

    if (result && result != GLOB_NOMATCH)
      errorExit();

    for (size_t i = 0; i < gt.gl_pathc; ++i) {
      const size_t len = strlen(gt.gl_pathv[i]);
      if (len && gt.gl_pathv[i][len-1] != '/')
        sfvList.push_back(gt.gl_pathv[i]);
    } // end for each globbed file
  } // end if no files specified on command line

  for (int i = 1; i < argc; ++i) {
      sfvList.push_back(argv[i]);
  } // end for argv
#endif
  //..................................................................

  if (sfvList.empty()) {
    fputs("\nDidn't find any SFV or MD5 files to check.\n", stderr);
    usage(false, exUsage);
  }
} // end VerifyFileList::VerifyFileList

//--------------------------------------------------------------------
// Summarize the result for our exit status:

int VerifyFileList::getExitStatus() const
{
  if (total[cntIOError])  return exIOError;
  if (total[cntMismatch]) return exMismatch;
  if (total[cntMissing])  return exMissing;

  return exSuccess;
} // end VerifyFileList::getExitStatus

//--------------------------------------------------------------------
// Open the next file to be verified:
//
// Output:
//   file:      The file to process
//   step:      The value to increment the progress counter by
//   interval:  The number of reads between progress updates
//   wantCRC:   True to calculate CRC32, false to calculate MD5
//
// Returns:
//   true:   Opened a new file
//   false:  No more files to process

bool VerifyFileList::getFile(fileT& file, int& step, int& interval,
                             bool& wantCRC)
{
  if (fileList.empty()) goto first_file; // We must be just starting out

 retry:
  if (++curFile == fileList.end()) {
    printResult(false);         // Show subtotals
    {
      for (int i = 0; i < cntArraySize; ++i) {
        total[i] += cnt[i];
        cnt[i]    = 0;
      }
    }
   first_file:
    if (!openNextSFV(fileList.empty()))
      return false;
  } // end if need new SFV file

  const char* filename = curFile->c_str();

  filename += checkLen[*filename - 1] + 1;

  printFile(filename);
  ++cnt[cntFiles];
  if (!wantProgress)
    for (int i = 72 - 2*curFileWidth - strlen(filename); i > 0; --i)
      putchar('.');

#ifdef _WIN32
  file = CreateFile(
    filename, GENERIC_READ, shareMode, NULL,
    OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL
  );
#else
  file = open(filename, O_RDONLY);
#endif

  if (file == invalidFile) {
    switch (LAST_ERROR) {
#ifdef _WIN32
     case ERROR_FILE_NOT_FOUND:
     case ERROR_PATH_NOT_FOUND:
#else
     case ENOENT:
#endif
      puts("MISS");
      ++cnt[cntMissing];
      break;

     default:
      puts("ERR ");
      printError();
      ++cnt[cntIOError];
      break;
    } // end switch GetLastError()
    goto retry;
  } // end if file couldn't be opened

  calcInterval(file, step, interval);
  wantCRC = ((*curFile)[0] == checkCRC32);

  return true;
} // end VerifyFileList::getFile

//--------------------------------------------------------------------
// Read the list of files from the next SFV file:
//
// Input:
//   first:  True means to open the first SFV file
//
// Input Variables:
//   relativePaths
//
// Output Variables:
//   curSfv, fileList, curFile, curFileWidth
//
// Returns:
//   true:   Opened a new SFV file
//   false:  No more files to process

bool VerifyFileList::openNextSFV(bool first)
{
  char  filename[PATH_MAX + 11 + MD5hexlen];

  if (first)
    curSfv = sfvList.begin();
  else {
   next_file:
    fileList.clear();
    if (++curSfv == sfvList.end())
      return false;             // No more files
  }

  FILE*  in = fopen(curSfv->c_str(), "r");
  if (!in) errorExit(curSfv->c_str(), exIOError);

  if (sfvList.size() > 1)
    printf("\n[%d/%d] %s:\n", ++sfvCount, sfvList.size(), curSfv->c_str());
  else
    printf("\n%s:\n", curSfv->c_str());

  String todo, pathname, basePath;
  int    lineNum = 0;

  if (relativePaths) {
    StrIdx pos = curSfv->find_last_of(dirChars);
    if (pos != String::npos)
      basePath.assign(*curSfv, 0, pos+1);
  } // end if using relativePaths

  while (fgets(filename, sizeof(filename), in)) {

    ++lineNum;
    if (filename[0] == ';') continue; // Skip comments

    int len = strlen(filename);
    if (!len) break;
    if (filename[len-1] != '\n') {
      fprintf(stderr, "%s:%d: Line too long, skipped it\n",
              curSfv->c_str(), lineNum);
      ++cnt[cntIOError];
      while (len && (filename[len-1] != '\n') &&
             fgets(filename, sizeof(filename), in))
        len = strlen(filename);
      if (len && filename[len-1] == '\n') continue; // Read next line
      break; // Reached end of file
    } // end if didn't read complete line

    filename[--len] = 0;        // Delete newline
#ifndef _WIN32
    if (len && filename[len-1] == '\r')
      filename[--len] = 0;      // Delete CR (MS-DOS text file)

    // Translate DOS pathnames to Unix (change \ to /):
    char* s = filename;
    while (NULL != (s = strchr(s, '\\')))
      *s = '/';
#endif

    todo.erase();
    pathname.erase();

    if (strncasecmp("MD5", filename, 3) == 0) {
      const char* start = filename + 3;
      while (isSpace(*start))
        ++start;
      if ((*start != '(') ||
          ((len - (++start - filename)) < MD5hexlen + 5))
        goto guessType;

      const char* md5 = filename + (len - MD5hexlen);
      const char* end = md5-1;
      while ((end != start) && isSpace(*end))
        --end;
      if ((end == start) || (*(end--) != '=')) goto guessType;
      while ((end != start) && isSpace(*end))
        --end;
      if ((end == start) || (*end != ')')) goto guessType;

      len = end - start;
      todo = checkMD5;
      todo.append(md5, MD5hexlen);
      pathname.assign(start, len);
    } // end if MD5

   guessType:
    // Try a md5sum line (eg: "0123456789abcdef0123456789abcdef  filename")
    if (todo.empty() &&
        // Begins with the right number of hex digits
        (strspn(filename, hexDigits) == size_t(MD5hexlen)) &&
        // Followed by space or tab
        isSpace(filename[MD5hexlen]) &&
        // Followed by space or '*' (binary file indicator)
        (filename[MD5hexlen+1] == ' ' || filename[MD5hexlen+1] == '*') &&
        // Followed by filename
        filename[MD5hexlen+2]) {
      todo = checkMD5;
      todo.append(filename, MD5hexlen);
      pathname.assign(filename + MD5hexlen + 2);
    } // end if found what looks like an md5sum line

    // Assume it's an SFV line (eg: "filename 1A2B3C4D")
    if (todo.empty()) {
      if ((len < 10) ||
          !isSpace(filename[len-CRChexlen-1]) ||
          strspn(filename + (len-CRChexlen), hexDigits) != size_t(CRChexlen)) {
       skipIt:
        const char* s = filename;
        while (isSpace(*s))
          ++s;
        if (*s) {
          fprintf(stderr, "%s:%d: Could not understand line, skipped it\n",
                  curSfv->c_str(), lineNum);
          ++cnt[cntIOError];
        } // end if the line was not blank, print error
        continue;
      } // end if not SFV line

      todo = checkCRC32;
      todo.append(filename + (len-CRChexlen), CRChexlen);
      for (len -= CRChexlen+1; len; --len)
        if (!isSpace(filename[len]))
          break;
      if (!len && isSpace(*filename)) goto skipIt;
      pathname.assign(filename, len+1);
    } // end if haven't found file to check yet

    if (pathname.empty()) {
      // This ought not to be possible, but better safe than sorry:
      fprintf(stderr, "%s:%d: No filename in line, skipped it\n",
              curSfv->c_str(), lineNum);
      ++cnt[cntIOError];
    } else {
      if (relativePaths && isRelativePath(pathname))
        todo.append(basePath);

      todo.append(pathname);
      fileList.push_back(todo);
    } // end else we found a pathname
  } // end while not eof

  fclose(in);

  if (fileList.empty())
    goto next_file;

  curFile = fileList.begin();
  calcFileWidth();

  return true;
} // end VerifyFileList::openNextSFV

//--------------------------------------------------------------------
// Print result for the current file:
//
// Input:
//   digest:  The checksum for the file

void VerifyFileList::setResult(const char* digest)
{
  const char*  expected = curFile->c_str();

  if (!strncasecmp(digest, expected+1, checkLen[*expected - 1]))
    puts(" OK ");
  else {
    puts("BAD ");
    ++cnt[cntMismatch];
  }

  if (curError) {
    printError(curError);
    curError = 0;
  }
} // end VerifyFileList::setResult

//--------------------------------------------------------------------
// Print a line of output:
//
// The line is omitted if count is 0.  The third printf field is
// assumed to be %s, and is supplied with "" if count is 1, and "s"
// otherwise.
//
// Input:
//   format:  The printf format string to use
//   width:   The first printf field (the field width for count)
//   count:   The second printf field

void VerifyFileList::printResult(const char* format, int width, int count) const
{
  if (count)
    printf(format, width, count, (count == 1 ? "" : "s"));
} // end VerifyFileList::printResult

//--------------------------------------------------------------------
// Print results after processing a SFV file:
//
// Input:
//   showTotals:  If true, print totals for all SFV files

void VerifyFileList::printResult(bool showTotals)
{
  if (showTotals) {
    if (sfvCount < 2) return;   // No totals unless multiple files
    putchar('\n');
    memcpy(cnt, total, sizeof(cnt));
  }

  int width = calcWidth(cnt[cntFiles]);

  printResult("%*d file%s checked",     width, cnt[cntFiles]);
  if (showTotals) {
    printf(" in %d SFV files", sfvCount);
    printResult(", found errors in %*d SFV file%s", 1, sfvBadCount);
  }

  printResult("\n%*d file%s missing",   width, cnt[cntMissing]);
  printResult("\n%*d file%s different", width, cnt[cntMismatch]);
  printResult("\n%*d other error%s",    width, cnt[cntIOError]);

  if (!cnt[cntMissing] && !cnt[cntMismatch] && !cnt[cntIOError])
    puts(", all ok!");
  else {
    putchar('\n');
    ++sfvBadCount;
  }
} // end VerifyFileList::printResult

//====================================================================

//====================================================================
// Initialization and option processing:
//====================================================================
// Display license information and exit:

bool license(GetOpt*, const GetOpt::Option*, const char*,
             GetOpt::Connection, const char*, int*)
{
  puts(titleString);
  puts("\n"
"This program is free software; you can redistribute it and/or\n"
"modify it under the terms of the GNU General Public License as\n"
"published by the Free Software Foundation; either version 2 of\n"
"the License, or (at your option) any later version.\n"
"\n"
"This program is distributed in the hope that it will be useful,\n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
"GNU General Public License for more details.\n"
"\n"
"You should have received a copy of the GNU General Public License\n"
"along with this program; if not, write to the Free Software\n"
"Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA."
  );

  exitNow(exSuccess);
  return false;                 // Never happens
} // end license

//--------------------------------------------------------------------
// Display version & usage information and exit:
//
// Input:
//   showHelp:    True means display usage information
//   exitStatus:  Status code to pass to exit()

void usage(bool showHelp, int exitStatus)
{
  FILE*  out = (exitStatus ? stderr : stdout);

  fputs(titleString, out);
  fputc('\n', out);

  if (showHelp) {
      fputs(
"\nUsage: FFV [options] [FILE ...]\n"
"  -a, --append          Append to existing checksum file (when creating one)\n"
"  -o, --overwrite       Replace an existing checksum file (when creating one)\n"
"  -e, --allow-errors    Don't stop if there's an error reading a file\n"
"  -h, --hidden          Include hidden files when creating a checksum file\n"
"  -m, --create-md5=FILE Create an MD5 checksum file\n"
"  -s, --create-sfv=FILE Create a SFV (CRC-32) file\n"
"  -n, --no-verify       Just check if files exist (don't verify checksums)\n"
"  -p, --pause           Wait for keypress before exiting\n"
"  -r, --relative        Interpret paths relative to the checksum file\n"
#ifdef _WIN32
"  -w, --allow-writing   Process files that are being written by other programs\n"
#endif
"  -?, --help            Display this help message\n"
"      --license         Display license information\n"
"      --version         Display version information\n\n"
"Note: You can't combine the --create-sfv and --create-md5 options\n"
"      (or the --append and --overwrite options, either).\n",
        out
      );
  } // end if showHelp

  exitNow(exitStatus);
} // end usage

bool usage(GetOpt* getopt, const GetOpt::Option* option,
           const char*, GetOpt::Connection, const char*, int*)
{
  usage(option->shortName == '?');
  return false;                 // Never happens
} // end usage

//--------------------------------------------------------------------
// Handle options:
//
// Input:
//   argc, argv:  The parameters passed to main
//
// Output:
//   argc, argv:
//     Modified to list only the non-option arguments
//     Note: argv[0] may not be the executable name

GetOpt::Found  md5Output, sfvOutput, existOnly;
IfExist        ifExist = iePrompt;
const char*    outputFileName = NULL;

bool appendOpt(GetOpt* getopt, const GetOpt::Option* option,
               const char*, GetOpt::Connection, const char*, int*)
{
  if (ifExist)
    getopt->reportError("", "You cannot specify more than one of "
                        "--append and --overwrite");

  if (option->shortName == 'o')
    ifExist = ieOverwrite;
  else
    ifExist = ieAppend;

  return false;                 // No argument
} // end appendOpt

void processOptions(int& argc, char**& argv)
{
#ifdef _WIN32
  GetOpt::Found  allowWriting;
#endif
  GetOpt::Found  allowErrors;

  static const GetOpt::Option options[] =
  {
    { 'a', "append",     NULL, 0, &appendOpt },
    { 'o', "overwrite",  NULL, 0, &appendOpt },
    { 'e', "allow-errors", &allowErrors },
    { 'h', "hidden",     &includeHidden },
    { 'm', "create-md5", &md5Output, GetOpt::needArg, &GetOpt::isString,
                         &outputFileName },
    { 's', "create-sfv", &sfvOutput, GetOpt::needArg, &GetOpt::isString,
                         &outputFileName },
    { 'n', "no-verify",  &existOnly },
    { 'p', "pause",      &pauseAtQuit },
    { 'r', "relative",   &relativePaths },
#ifdef _WIN32
    { 'w', "allow-writing", &allowWriting },
#endif
    { '?', "help",       NULL, 0, &usage },
    { 0,   "license",    NULL, 0, &license },
    { 0,   "version",    NULL, 0, &usage },
    { 0 }
  };

  GetOpt getopt(options);
  int argi = getopt.process(argc, const_cast<const char**>(argv));
  if (getopt.error || (sfvOutput && md5Output) ||
      ((sfvOutput || md5Output) && (!outputFileName || !*outputFileName)))
    usage(true, exUsage);

  if (md5Output || sfvOutput) {
    if (existOnly) {
      fputs("You cannot specify --no-verify when creating a checksum file\n",
            stderr);
      usage(true, exUsage);
    }
    if (relativePaths) {
      fputs("You cannot specify --relative when creating a checksum file\n",
            stderr);
      usage(true, exUsage);
    }
  } // end if creating a checksum file

  if (allowErrors)  stopOnError = false;
#ifdef _WIN32
  if (allowWriting) shareMode = FILE_SHARE_READ | FILE_SHARE_WRITE;
#endif

  if (argi >= argc)
    argc = 1;           // No arguments
  else {
    argc -= --argi;     // Reduce argc by number of arguments used
    argv += argi;       // And adjust argv[1] to the next argument
  }
} // end processOptions

//====================================================================
// Main Program:
//====================================================================

#ifdef _WIN32
#include "win32/verify.cpp"
#else
#include "posix/verify.cpp"
#endif

//--------------------------------------------------------------------
// Just check if the files listed in a checksum file can be opened:

void checkExist(int argc, char* argv[])
{
  puts("\nChecking only to see if files exist and can be opened!");
  VerifyFileList fileList(argc, argv);

  fileT   inFile;
  int     step, interval;
  bool    found = false;
#ifndef _WIN32
  bool    wantCRC = false;
#endif

  while (fileList.getFile(inFile, step, interval, wantCRC)) {
    close(inFile);
    puts(" OK ");
    found = true;
  } // end while opened another file

  fileList.printResult();

  if (found)
    puts("Note: The file contents were not verified!");

  exitNow(fileList.getExitStatus());
} // end checkExist

//--------------------------------------------------------------------
int main(int argc, char* argv[])
{
  processOptions(argc, argv);
  puts(titleString);

  if (isatty(fileno(stdout))) {
    wantProgress = true;
#ifndef _WIN32
    // Turn buffering off so we can see the progress indicator:
#ifdef SETVBUF_REVERSED
    setvbuf(stdout, _IONBF, NULL, 0);
#else
    setvbuf(stdout, NULL, _IONBF, 0);
#endif
#endif
  } // end if stdout is a tty

  if (existOnly)
    checkExist(argc, argv);
  else
    createOrVerify(argc, argv);

  return 0;                     // Never happens
} // end main

// Local Variables:
//  c-file-style: "cjm"
// End:
