//--------------------------------------------------------------------
// $Id: verify.cpp 4682 2007-01-25 05:10:59Z cjm $
//--------------------------------------------------------------------
//
//   Fast File Validator
//   Copyright 2004 by Christopher J. Madsen
//
//   File verification (POSIX version)
//
//   This program is free software; you can redistribute it and/or
//   modify it under the terms of the GNU General Public License as
//   published by the Free Software Foundation; either version 2 of
//   the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program; if not, write to the Free Software
//   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//--------------------------------------------------------------------

//====================================================================
// Create or verify a checksum file:

#include "../crc32.h"

void createOrVerify(int argc, char* argv[])
{
  DiskBuffer* buffer = reinterpret_cast<DiskBuffer*>(valloc(bufferSize*2));

  if (!buffer) {
    fputs("Failed to allocate buffer!\n", stderr);
    exitNow(exInternalError);
  }

  struct aiocb  a;
  const struct aiocb *const aiobl[] = { &a };
  memset(&a, 0, sizeof(a));
  a.aio_sigevent.sigev_notify = SIGEV_NONE;

  size_t bytesRead;
  DWORD  crc = 0xFFFFFFFF;
  MD5    md5;
  int    bufferNum = 0;
  bool   fileUnfinished;
  bool   wantCRC = false;
  char   outputBuf[MD5hexlen+1];

  FileList* sfv;
  if (md5Output || sfvOutput)
    sfv = new CreateFileList(outputFileName, ifExist, sfvOutput != 0,
                             argc, argv);
  else
    sfv = new VerifyFileList(argc, argv);

  int    step, interval, complete = 0;
  if (!sfv->getFile(a.aio_fildes, step, interval, wantCRC))
    exitNow(sfv->getExitStatus());
  int count = interval;

  a.aio_offset = 0;
  a.aio_buf    = buffer[bufferNum];
  a.aio_nbytes = bufferSize;

  if (aio_read(&a)) {
    if (stopOnError) errorExit(sfv->currentFile(), exIOError);
    sfv->recordError();
    goto nextFile;
  }

  for (;;) {
    if (aio_suspend(aiobl, 1, NULL) ||
        (0 != (errno = aio_error(&a)))) {
      if (stopOnError) errorExit(sfv->currentFile(), exIOError);
      sfv->recordError();
      goto nextFile;
    } // end if the I/O operation failed

    bytesRead = aio_return(&a);

    fileUnfinished = (bytesRead == bufferSize);
    if (fileUnfinished) {
      a.aio_offset += bufferSize;
      a.aio_buf    = buffer[!bufferNum];
      a.aio_nbytes = bufferSize;
      if (aio_read(&a)) {
        if (stopOnError) errorExit(sfv->currentFile(), exIOError);
        sfv->recordError();
        fileUnfinished = false; // Move on to the next file
      } // end if aio_read failed
    } // end if fileUnfinished

    if (wantProgress && !--count) {
      count = interval;
      printf("%3d\r", (complete += step) / pci);
    }

    if (wantCRC) {
      register BYTE*  buf = buffer[bufferNum];
      for (register int len = bytesRead; len; --len)
        crc = crc_32_tab[(crc ^ *(buf++)) & 0xFF] ^ (crc >> 8);
    } else
      md5.update(buffer[bufferNum], bytesRead);

    if (fileUnfinished)
      bufferNum = !bufferNum;
    else { // End of file
     nextFile:
      close(a.aio_fildes);
      if (wantProgress && !sfv->currentError()) fputs("100%\r", stdout);
//      printf("%3d %d %d ", complete/pci, step, interval);

      if (wantCRC) {
        sprintf(outputBuf, "%08lX", ~crc);
        crc = 0xFFFFFFFF;
      } else
        hexdigest(outputBuf, md5); // Also reinitializes md5

      sfv->setResult(outputBuf);

      if (!sfv->getFile(a.aio_fildes, step, interval, wantCRC))
        break;                  // No more files
      complete = 0;
      count    = interval;

      a.aio_offset = 0;
      a.aio_buf    = buffer[bufferNum];
      a.aio_nbytes = bufferSize;

      if (aio_read(&a)) {
        if (stopOnError) errorExit(sfv->currentFile(), exIOError);
        sfv->recordError();
        goto nextFile;
      } // end if aio_read failed
    } // end if end of file
  } // end forever

  sfv->printResult();
  const int  status = sfv->getExitStatus();

  delete sfv;
  free(buffer);

  exitNow(status);
} // end createOrVerify
